/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 *****************************************************************************/


/* ---------------------------------------------------------------
 * File: xser32.c 
 * -----------------------------------------------------------------*/

#include <windows.h>
#include <assert.h>

#include <xtypedef.h>

#include <xio.h>
#include <xserial.h>
#include <xlasterr.h>
#include <xutil.h>

/* -------------------------------------------------------------------------
 * STATICS
 * ------------------------------------------------------------------------- */

static void PrintErrorMessage (const char * fnName, DWORD errCode);

/*****************************************************************************/
static bx_errtype BestXClearCommError(bx_portnumtype OsHandle)
{
  unsigned long lErrors = 0UL;
  int MaxTries = 10;
  COMSTAT ComStat;

  /* Note; Do not use FlushFileBuffers(),
   * it will transmit the rest of the output buffer !
   */

  /* may have to call this multiple times for breaks */
  do
  {
    (void) ClearCommError((HANDLE) OsHandle, &lErrors, &ComStat);
  } while (lErrors && --MaxTries);

  /* purge regardless */
  (void) PurgeComm((HANDLE) OsHandle, 
    (PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR));

  /* return a generic error.  
   * CONSIDER; We could parse and return better info.
   */

  return BX_E_ERROR;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXDeviceConnect()
 * We set our DTR (card's DSR) to signal connect.
 * ------------------------------------------------------------------------- */

bx_errtype BestXSerDeviceConnect(bx_portnumtype OsHandle)
{
  if(OsHandle == INVALID_OS_HANDLE)
    return BX_E_BAD_HANDLE;

  /* start with a clean slate */
  (void) BestXClearCommError(OsHandle);
  (void) EscapeCommFunction((HANDLE) OsHandle, SETDTR);
  /*   DBG_PRINT("Serial Port Connected\n"); */

  return BX_E_OK;
}

/* --------------------------------------------------------------------------
 * Should only be called from BestXCheckConnection()
 * Card sets its DTR (our DSR) to signal connected.
 * ------------------------------------------------------------------------- */

bx_errtype BestXSerCheckConnection(bx_portnumtype OsHandle)
{
  DWORD dwRegVal = 0; /* MS_DSR_ON; */

  if(OsHandle == INVALID_OS_HANDLE)
    return BX_E_BAD_HANDLE;

  if( GetCommModemStatus((HANDLE) OsHandle, &dwRegVal) )
  {
    /* DSR high ? */
    if(MS_DSR_ON & dwRegVal)
      return BX_E_OK;
  }
  else
  {
    /* This should NEVER happen (mixing baud rates will do it!)
     * Treat as a serious problem but DON'T disconnect.
     */
    DBG_ApiFailureIsZero(0);
    BestXSerOnReadOrWriteError(OsHandle);
  }

  return BX_E_NOT_CONNECTED;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXReleaseConnection()
 * We clear our DTR (card's DSR) to signal disconnect.
 * ------------------------------------------------------------------------- */

void BestXSerReleaseConnection(bx_portnumtype OsHandle)
{
  if(OsHandle == INVALID_OS_HANDLE)
    return;

  (void) EscapeCommFunction((HANDLE) OsHandle, CLRDTR);
  /*   DBG_PRINT("Serial Port Disconnected\n"); */
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXOnReadError()
 * ------------------------------------------------------------------------- */

bx_errtype BestXSerOnReadOrWriteError(bx_portnumtype OsHandle)
{
#ifdef BEST_DEBUG

/* DEBUG ONLY ... display the real error */

  DCB dcb;
  COMMTIMEOUTS touts;
  unsigned long lErrors;
  COMSTAT ComStat;
  char buf[200];

  /* GetCommTimeouts() returns 0 on failure */
  if(GetCommTimeouts((HANDLE) OsHandle, &touts))
  {
    BESTX_SPRINTF(buf, "\nCurrent Timeouts: %d %d %d %d %d", 
      touts.ReadIntervalTimeout,
      touts.ReadTotalTimeoutMultiplier,
      touts.ReadTotalTimeoutConstant,
      touts.WriteTotalTimeoutMultiplier,
      touts.WriteTotalTimeoutConstant
      );
  }
  else
  {
    BESTX_STRCPY(buf, "\n Failure getting Win API Current Timeouts");
  }

  DBG_PRINT(buf);

  /* can break here if required */
  (void) GetCommState((HANDLE) OsHandle, &dcb);

  (void) ClearCommError((HANDLE) OsHandle, &lErrors, &ComStat);

  if(lErrors)
  {
    if(lErrors & CE_RXOVER)   DBG_PRINT(" Comm Err *** Receive Queue overflow\n");
    if(lErrors & CE_OVERRUN)  DBG_PRINT(" Comm Err *** Receive Overrun Error\n");
    if(lErrors & CE_RXPARITY) DBG_PRINT(" Comm Err *** Receive Parity Error\n");
    if(lErrors & CE_FRAME)    DBG_PRINT(" Comm Err *** Receive Framing error\n");
    if(lErrors & CE_BREAK)    DBG_PRINT(" Comm Err *** Break Detected\n");
    if(lErrors & CE_TXFULL)   DBG_PRINT(" Comm Err *** TX Queue is full\n");
    if(lErrors & CE_TXFULL)   DBG_PRINT(" Comm Err *** Requested mode unsupported\n");
  }

  /* COMSTAT structure contains information regarding
   * communications status.    
   */
  if (ComStat.fCtsHold)  DBG_PRINT(" ComStat *** Tx waiting for CTS signal\n");
  if (ComStat.fDsrHold)  DBG_PRINT(" ComStat *** Tx waiting for DSR signal\n");
  if (ComStat.fRlsdHold) DBG_PRINT(" ComStat *** Tx waiting for RLSD signal\n");
  if (ComStat.fXoffHold) DBG_PRINT(" ComStat *** Tx waiting, XOFF char rec'd\n");
  if (ComStat.fXoffSent) DBG_PRINT(" ComStat *** Tx waiting, XOFF char sent\n");
  if (ComStat.fEof)      DBG_PRINT(" ComStat *** EOF character received\n");
  if (ComStat.fTxim)     DBG_PRINT(" ComStat *** Character waiting for Tx; char queued with TransmitCommChar\n");
  if (ComStat.cbInQue)   DBG_PRINT(" ComStat *** byte(s) have been received, but not read\n");
  if (ComStat.cbOutQue)  DBG_PRINT(" ComStat *** byte(s) are awaiting transfer\n");

  (void) PurgeComm((HANDLE) OsHandle, 
    (PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR));

  return BX_E_OK;

#else

  if(OsHandle == INVALID_OS_HANDLE)
    return BX_E_BAD_HANDLE;

  return BestXClearCommError(OsHandle);

#endif
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXPortTimeoutSet()
 * ------------------------------------------------------------------------- */
bx_errtype BestXSerPortTimeoutSet(bx_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  return SetCommTimeouts(OsHandle, (LPCOMMTIMEOUTS) pCallersTimeouts) ?
          BX_E_OK : 
          BX_E_SETTING_TIMEOUTS;
}


/*****************************************************************************/

#define SER_IN_BUF_SIZE 4096
#define SER_OUT_BUF_SIZE 4096

/*---------------------------------------------------------------------------*
 * bx_portnumtype BestXOpenCOMInitial(int num, bx_int32 baudrate)
 *
 * Purpose: open connection to COM port (driver!) and initialize
 *                NOTE: no connection to device is established yet!!!
 * Inputs: portnum (1 == COM1, 2 == COM2, etc.), baudrate
 * Outputs: none
 * Returns: OS Handle
 *---------------------------------------------------------------------------*/
bx_portnumtype BestXOpenCOMInitial(int num, bx_int32 baudrate)
{
  HANDLE OsHandle;
  LPCTSTR lpComPort;
  char comPortStr[6]; /* "COM" + double digit + '\0' ! */

  /* check for valid port number */
  if (num < 1 && num > 99)
  {
    return INVALID_OS_HANDLE;
  }

  sprintf (comPortStr, "COM%d", num);
  lpComPort = (LPCTSTR) comPortStr;

  /* open the comport as a file */
  OsHandle = CreateFile( lpComPort,
                         GENERIC_READ | GENERIC_WRITE,
                         0,         /* no sharing */
                         NULL,      /* security attributes */
                         OPEN_EXISTING,
                         0,         /* file attributes */
                         NULL );

  if (OsHandle == INVALID_HANDLE_VALUE) /* convert from OS error to our one */
  {
    PrintErrorMessage ("BestXOpenCOMInitial", GetLastError());
    return INVALID_OS_HANDLE;
  }

  if (BestXSerInit(OsHandle, baudrate) == BX_E_OK)
  {
    /* just to get rid of 'hanging old' connections to devices */
    BestXSerReleaseConnection((bx_portnumtype) OsHandle);
  }
  else
  {
    OsHandle = INVALID_OS_HANDLE;
  }
  
  return (bx_portnumtype) OsHandle;
}

/*---------------------------------------------------------------------------*
 * void BestXSerInit (bx_portnumtype OsHandle, bx_int32 baudrate)
 *
 * Purpose: Initialize COM port settings
 * Inputs: OS Handle, baudrate
 * Outputs: 
 * Returns: bx_errtype
 *---------------------------------------------------------------------------*/
bx_errtype BestXSerInit (bx_portnumtype OsHandle, bx_int32 baudrate)
{
  DWORD brate;  /* baudrate in WIN32 format */
  DCB dcb;      /* device control block, see Win32 docu */

  /* check the baudrate */
  switch (baudrate)
  {
   case 2400:
    brate = CBR_2400;
    break;
   case 4800:
    brate = CBR_4800;
    break;
   case 9600:
    brate = CBR_9600;
    break;
  case 19200:
    brate = CBR_19200;
    break;
  case 38400:
    brate = CBR_38400;
    break;
  case 57600:
    brate = CBR_57600;
    break;
  case 115200:
    brate = CBR_115200;
    break;
  case 124800:
    brate = CBR_128000;
    break;
  default:
    BestXCloseSerial(OsHandle);
    return BX_E_BAUDRATE;
  }

  /* No monitoring */
  (void) SetCommMask(OsHandle, 0);

  /* buffer sizes */
  (void) SetupComm(OsHandle, SER_IN_BUF_SIZE, SER_OUT_BUF_SIZE);

  /* setup protocol and line control.
   * DTR (from PC) and DSR (from Card) are used to indicate "connected".
   * RTS/CTS used for hardware flow control.  
   * DSR sensitivity is used to help prevent junk in the input buffer.
   */
  (void) GetCommState(OsHandle, &dcb);

  dcb.DCBlength = sizeof(DCB);
  dcb.BaudRate = brate;
  dcb.fBinary = TRUE;
  dcb.fParity = FALSE;
  dcb.fOutxCtsFlow = TRUE;
  dcb.fOutxDsrFlow = TRUE;  /* was false */
  dcb.fDtrControl = DTR_CONTROL_DISABLE;
  dcb.fDsrSensitivity = TRUE;
  dcb.fOutX = FALSE;
  dcb.fInX = FALSE;
  dcb.fRtsControl = RTS_CONTROL_HANDSHAKE;
  dcb.fAbortOnError = FALSE;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.StopBits = ONESTOPBIT;

  if (!SetCommState(OsHandle, &dcb))
  {
    BestXCloseSerial(OsHandle);
    return BX_E_NOT_CONNECTED;  /* TODO: find good error code here! */
  }

  return BX_E_OK;
}

/*****************************************************************************/
void BestXCloseSerial(bx_portnumtype OsHandle)
{
  BestXSerReleaseConnection(OsHandle);
  if (!CloseHandle((HANDLE) OsHandle))
  {
    PrintErrorMessage ("BestXCloseSerial", GetLastError());
  }
}

/*---------------------------------------------------------------------------*
 * static void PrintErrorMessage (const char * fnName, DWORD errCode)
 *
 * Purpose: get error info from NT
 *---------------------------------------------------------------------------*/
static void PrintErrorMessage (const char * fnName, DWORD errCode)
{
    LPVOID lpMsgBuf;
    char msgBuf[512];
    
    FormatMessage( 
      FORMAT_MESSAGE_ALLOCATE_BUFFER | 
      FORMAT_MESSAGE_FROM_SYSTEM | 
      FORMAT_MESSAGE_IGNORE_INSERTS,
      NULL,
      errCode,
      MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
      (LPTSTR) &lpMsgBuf,
      0,
      NULL 
      );
    // Process any inserts in lpMsgBuf.
    // ...
    // Display the string.
    sprintf (msgBuf, "%s returned (%ld) %s\n", fnName, errCode, 
      (const char *) lpMsgBuf);

    DBG_PRINT(msgBuf);
    // Free the buffer.
    LocalFree( lpMsgBuf );
}
